<?php
namespace Tygh\Api\Entities;

require_once dirname(__FILE__) . '/../../../TestUnitDB.php';

use Tygh\Api\Entities\Languages;
use Tygh\Api\Response;
/**
 * Tests core API functionality via CURL
 */
class LanguagesTest extends \TestUnitDB
{
    /**
     * @var \Tygh\Api\Entities\Languages
     */
    protected $api_languages;

    protected function setUp()
    {
        parent::setUp();

        $this->api_languages = new Languages(array(
            'area' => 'A',
            'user_id' => '1',
            'user_type' => 'A',
            'tax_exempt' => 'N',
            'last_login' => '1343637055',
            'usergroup_ids' => array ( ),
            'order_ids' => array ( ),
            'act_as_user' => false,
            'this_login' => 1343652763,
            'password_change_timestamp' => 0,
            'company_id' => 0,
            'is_root' => 'Y',
            'referer' => '',
            'ip' => '192.168.1.111',
            'first_expire_check' => false
        ));

    }

    protected function tearDown()
    {
    }

    protected function languagesDataset()
    {
        return array(
            'languages.xml',
        );
    }

    /**
     *
     * @dataProvider providerIndex()
     * @datasetProvider languagesDataset
     */
    public function testIndex($id, $params, $result)
    {
        $this->assertEquals($result, $this->api_languages->index($id, $params));
    }

    public function providerIndex()
    {
        return array(
            // Get languages list
            array('', array(), array(
                'status' => Response::STATUS_OK,
                'data' => array(
                    'languages' => array(
                        'en' => array(
                            'lang_id' => 1,
                            'lang_code' => 'en',
                            'name' => 'English',
                            'status' => 'A',
                            'country_code' => 'US',
                        ),
                        'ru' => array(
                            'lang_id' => 2,
                            'lang_code' => 'ru',
                            'name' => 'Russian',
                            'status' => 'A',
                            'country_code' => 'RU',
                        ),
                    ),

                    'params' => array(
                        'items_per_page' => '10',
                        'page' => 1,
                        'total_items' => 2,
                    ),
                ),
            )),

            // Get language
            array(1, array(), array(
                'status' => Response::STATUS_OK,
                'data' => array(
                    'lang_id' => 1,
                    'lang_code' => 'en',
                    'name' => 'English',
                    'status' => 'A',
                    'country_code' => 'US',
                ),
            )),

            // Get language by code
            array('', array('lang_code' => 'en'), array(
                'status' => Response::STATUS_OK,
                'data' => array(
                    'lang_id' => 1,
                    'lang_code' => 'en',
                    'name' => 'English',
                    'status' => 'A',
                    'country_code' => 'US',
                ),
            ))
        );
    }

    /**
     *
     * @dataProvider providerCreate()
     * @datasetProvider languagesDataset
     */
    public function testCreate($params, $result)
    {
        $actual = $this->api_languages->create($params);
        $this->assertEquals($result, $actual);
    }

    public function providerCreate()
    {
        return array(
            // Try create with empty data
            array(array(), array(
                'status' => Response::STATUS_BAD_REQUEST,
                'data' => array(),
            )),
            // Try create without lang code
            array(array(
                'name' => 'Test',
                'status' => 'A',
                'country_code' => 'US'
            ), array(
                'status' => Response::STATUS_BAD_REQUEST,
                'data' => array(),
            )),
            // Try create
            array(array(
                'lang_code' => 'tt',
                'name' => 'Test',
                'status' => 'A',
                'country_code' => 'US'
            ), array(
                'status' => Response::STATUS_CREATED,
                'data' => array(
                    'lang_id' => 3,
                ),
            )),
        );
    }

    /**
     *
     * @dataProvider providerUpdate()
     * @datasetProvider languagesDataset
     */
    public function testUpdate($id, $params, $updated_language, $result)
    {
        $actual = $this->api_languages->update($id, $params);

        if ($actual['status'] == Response::STATUS_OK) {
            $res = $this->api_languages->index($actual['data']['lang_id']);
            $this->assertEquals($updated_language, $res['data']);
        } else {
            $this->assertEquals($result, $actual);
        }
    }

    public function providerUpdate()
    {
        return array(
            // update with empty id
            array('',
                array(
                    'lang_code' => 'tt',
                    'name' => 'Test',
                    'country_code' => 'TT'
                ),
                array(
                    'lang_id' => '3',
                    'lang_code' => 'tt',
                    'name' => 'Test',
                    'country_code' => 'TT',
                    'status' => 'A',
                ),
                array(
                    'status' => Response::STATUS_BAD_REQUEST,
                    'data' => array(),
                )
            ),

            // Try update. Bad id.
            array(3,
                array(
                    'lang_code' => 'tt',
                    'name' => 'Test',
                ),
                array(),
                array(
                    'status' => Response::STATUS_BAD_REQUEST,
                    'data' => array(),
                )
            ),

            // Try update
            array(1,
                array(
                    'lang_code' => 'tt',
                    'name' => 'Test',
                    'country_code' => 'TT',
                ),
                array(
                    'lang_id' => 1,
                    'lang_code' => 'tt',
                    'name' => 'Test',
                    'country_code' => 'TT',
                    'status' => 'A',
                ),
                array(
                    'status' => Response::STATUS_OK,
                    'data' => array(
                        'lang_id' => 1,
                    )
                )
            ),
        );
    }

    /**
     *
     * @dataProvider providerDelete()
     * @datasetProvider languagesDataset
     */
    public function testDelete($id, $result)
    {
        $actual = $this->api_languages->delete($id);
        $this->assertEquals($result, $actual);

        if ($actual['status'] == Response::STATUS_NO_CONTENT) {
            $actual = $this->api_languages->delete($id);
            $this->assertEquals($actual, array('status' => Response::STATUS_NOT_FOUND));
        }

    }

    public function providerDelete()
    {
        return array(
            // Try delete with empty id
            array('', array(
                'status' => Response::STATUS_NOT_FOUND,
                'data' => array()
            )),
            // Try delete with nonexistent id
            array(777, array(
                'status' => Response::STATUS_NOT_FOUND,
                'data' => array()
            )),
            // Try delete
            array(1, array(
                'status' => Response::STATUS_BAD_REQUEST,
                'data' => array()
            )),
        );
    }
}
